<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class ZoomController extends Controller
{
    public function __construct()
    {
        
        // we check if we have a token in the session if not we authorize the user
        if (!session('zoom_access_token')) {
            return redirect()->route('zoom.authorize');
        }
    }

    public function authorizeZoom()
    {
        $authorizeUrl = 'https://zoom.us/oauth/authorize?' . http_build_query([
            'response_type' => 'code',
            'client_id'     => env('ZOOM_CLIENT_ID'),
            'redirect_uri'  => env('ZOOM_REDIRECT_URI'),
        ]);

        return redirect($authorizeUrl);
    }

    public function handleZoomCallback(Request $request)
    {
        $code = $request->get('code');

        $response = Http::withHeaders([
            'Authorization' => 'Basic ' . base64_encode(env('ZOOM_CLIENT_ID') . ':' . env('ZOOM_CLIENT_SECRET')),
        ])->asForm()->post('https://zoom.us/oauth/token', [
            'grant_type'    => 'authorization_code',
            'code'          => $code,
            'redirect_uri'  => env('ZOOM_REDIRECT_URI'),
        ]);
        

        $data = $response->json();

        // Store access_token and refresh_token in DB or session
        session([
            'zoom_access_token' => $data['access_token'],
            'zoom_refresh_token' => $data['refresh_token'],
        ]);

        // Optionally, you can store the expiration time
        session([
            'zoom_token_expires_at' => now()->addSeconds($data['expires_in']),
        ]);
        $html =   '
        <h1>Zoom Authorization Successful</h1>
        <script>
            localStorage.setItem("zoom_token_updated", "'.now()->addSeconds($data['expires_in']).'"); 
            window.close();
        </script>
        <p>You can close this window now.</p>
        ';

        return response($html, 200)
        ->header('Content-Type', 'text/html')
        ->withCookie(cookie('zoom_access_token', $data['access_token'], 60))
        ->withCookie(cookie('zoom_refresh_token', $data['refresh_token'], 60))
        ->withCookie(cookie('zoom_token_expires_at', now()->addSeconds($data['expires_in']), 60));
        
    }

    public function createMeeting($startTime = null)
    {
        $accessToken = session('zoom_access_token');
       
        if (!$accessToken) {
            return redirect()->route('zoom.authorize');
        }
        // if startTime is not null we will use it
        // else we will use the current time + 10 minutes
        if ($startTime != null) {
            $time= $startTime;
        } else {
            $time = now()->addMinutes(10)->toIso8601String();
        }

        $response = Http::withToken($accessToken)
            ->post('https://api.zoom.us/v2/users/me/meetings', [
                'topic'      => 'Solveet Zoom Meeting',
                'type'       => 2,
                'start_time' => $time,
                'duration'   => 40,
                'timezone'   => 'Europe/Bucharest',
                'default_password' => false,
                'password'   => '',
                'settings'   => [
                    'join_before_host' => true,
                    'waiting_room'     => false,
                ]
            ]);
      
        $joinUrl = $response->json()['join_url'];
        $raspuns = [
            'join_url' => $joinUrl,
            'password' => $response->json()['password'],
        ];
        return $raspuns;
    }
}
