<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Google\Client as Google_Client;
use Google\Service\Calendar as Google_Service_Calendar;
use Google\Service\Calendar\Event as Google_Service_Calendar_Event;

class GoogleMeetController extends Controller
{
    protected function getGoogleClient()
    {
        $client = new Google_Client();
        $client->setClientId(env('GOOGLE_CLIENT_ID'));
        $client->setClientSecret(env('GOOGLE_CLIENT_SECRET'));
        $client->setRedirectUri(env('GOOGLE_REDIRECT_URI'));
        $client->setAccessType('offline');
        $client->setPrompt('consent');
        $client->setScopes([Google_Service_Calendar::CALENDAR_EVENTS]);
        return $client;
    }

    public function authorizeGoogle(Request $request)
    {
        $client = $this->getGoogleClient();
        $state = Str::random(40);
        session(['google_oauth_state' => $state]);
        $client->setState($state);

        $authUrl = $client->createAuthUrl();
        return redirect($authUrl);
    }

    public function handleGoogleCallback(Request $request)
    {
        $state = $request->get('state');
        if (!$state || $state !== session('google_oauth_state')) {
            abort(403, 'Invalid state');
        }
        session()->forget('google_oauth_state');

        $code = $request->get('code');
        if (!$code) {
            abort(400, 'Authorization code not provided');
        }

        $client = $this->getGoogleClient();
        $token = $client->fetchAccessTokenWithAuthCode($code);

        if (!isset($token['access_token'])) {
            abort(400, 'Failed to obtain access token');
        }

        session([
            'google_access_token'  => $token['access_token'],
            'google_refresh_token' => $token['refresh_token'] ?? session('google_refresh_token'),
            'google_token_expires_at' => now()->addSeconds($token['expires_in']),
        ]);

        $html = '
        <h1>Google Meet Authorization Successful</h1>
        <script>
            localStorage.setItem("google_token_updated", "' . now()->addSeconds($token['expires_in']) . '");
            window.close();
        </script>
        <p>You can close this window now.</p>
        ';

        return response($html, 200)
            ->header('Content-Type', 'text/html');
    }

    /**
     * Get a valid Google access token, refreshing if necessary.
     */
    protected function getValidAccessToken()
    {
        $client = $this->getGoogleClient();

        $accessToken = session('google_access_token');
        $refreshToken = session('google_refresh_token');
        $expiresAt = session('google_token_expires_at');

        if (!$accessToken || !$expiresAt) {
            return null;
        }

        $client->setAccessToken([
            'access_token' => $accessToken,
            'refresh_token' => $refreshToken,
            'expires_in' => $expiresAt->diffInSeconds(now(), false),
            'created' => now()->timestamp - ($expiresAt->diffInSeconds(now(), false) - $expiresAt->diffInSeconds(now())),
        ]);

        if ($client->isAccessTokenExpired() && $refreshToken) {
            $client->fetchAccessTokenWithRefreshToken($refreshToken);
            $newToken = $client->getAccessToken();
            if (isset($newToken['access_token'])) {
                session([
                    'google_access_token' => $newToken['access_token'],
                    'google_token_expires_at' => now()->addSeconds($newToken['expires_in']),
                ]);
                $accessToken = $newToken['access_token'];
            } else {
                session()->forget(['google_access_token', 'google_token_expires_at', 'google_refresh_token']);
                return null;
            }
        }

        return $client;
    }

    public function createMeeting($startTime = null)
    {
        $client = $this->getValidAccessToken();
        if (!$client) {
            return redirect()->route('google.authorize');
        }

        $calendarService = new Google_Service_Calendar($client);

        $time = $startTime ?? now()->addMinutes(10)->toIso8601String();
        $endTime = now()->addMinutes(50)->toIso8601String();

        $event = new Google_Service_Calendar_Event([
            'summary' => 'Solveet Google Meet Meeting',
            'start' => [
                'dateTime' => $time,
                'timeZone' => 'Europe/Bucharest',
            ],
            'end' => [
                'dateTime' => $endTime,
                'timeZone' => 'Europe/Bucharest',
            ],
            'conferenceData' => [
                'createRequest' => [
                    'requestId' => uniqid(),
                    'conferenceSolutionKey' => [
                        'type' => 'hangoutsMeet'
                    ]
                ]
            ],
        ]);

        $createdEvent = $calendarService->events->insert(
            'primary',
            $event,
            ['conferenceDataVersion' => 1]
        );

        $meetLink = null;
        if (isset($createdEvent->conferenceData->entryPoints)) {
            foreach ($createdEvent->conferenceData->entryPoints as $entryPoint) {
                if ($entryPoint->entryPointType === 'video') {
                    $meetLink = $entryPoint->uri;
                    break;
                }
            }
        }

        return [
            'meet_link' => $meetLink,
            'event_id'  => $createdEvent->id ?? null,
        ];
    }
}
